<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Constant;
use App\Filters\PostsFilter;
use App\Helpers\Format\Date;
use App\Helpers\Format\JsTree;
use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\File;
use App\Models\Post;
use App\Models\Tag;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;


class PostsController extends BaseController
{
    public function index()
    {
        $posts = Post::filter(new PostsFilter())->paginate();
        return view('admin.posts.all', compact('posts'));
    }

    public function create()
    {
        $statuses = Constant::getPostStatusesView();
        $categories = JsTree::getDataFor(Category::class, 'parent_id', ['name' => 'entity', 'value' => Constant::POST]);
        return view('admin.posts.create', compact('statuses', 'categories'));
    }

    public function store()
    {

        // validate request
        $this->validatePost();
        // prepare data to insert in db
        $data = $this->getData();
        // create new user
        $post = Post::create($data);
        // redirect
        if ($post instanceof Post) {
            //add tag
            $post->tags()->sync(request('tags'));
            //add category
            $post->categories()->sync(request('categories'));
            return redirect()->route('admin.posts.all')->with('success', 'ثبت پست با موفقیت انجام شد');
        }
        return redirect()->back();
    }

    public function edit(Post $post)
    {

        $statuses = Constant::getPostStatusesView();
        $selectedDataIDs = $post->categories()->get(['category_id as id'])->pluck('id')->toArray();
        $categories = JsTree::getDataFor(Category::class, 'parent_id', ['name' => 'entity', 'value' => Constant::POST], $selectedDataIDs);
        return view('admin.posts.edit', compact('post', 'statuses', 'categories'));
    }

    public function update(Post $post)
    {
        // validate request
        $this->validateupdateform($post);
        $data = $this->getData();
        $result = $post->update($data);
        if ($result) {
            //update tag
            $post->tags()->sync(request('tags'));
            //update category
            $post->categories()->sync(request('categories'));
            return redirect()->route('admin.posts.edit', $post)->with('success', 'پست شما با موفقیت ویرایش شد');
        }
        return redirect()->back();
    }

    public function delete(Post $post)
    {
        $post->delete();
        return redirect()->back()->with('delete', 'پست شما با موفقیت حدف شد');
    }

    public function comments(Post $post)
    {
        $comments = $post->comments;
        return view('admin.posts.comments', compact('post', 'comments'));
    }

    private function validatePost()
    {
        request()->validate([

            'description' => ['required'],
            'image' => ['required', 'image'],
            'tags' => ['required'],
            'categories' => ['required'],
            'title' => ['required'],
        ], [

            'description.required' => "توضیحات پست را وارد نمائید",
            'tags.required' => "  تگ پست را وارد نمائید",
            'categories.required' => "  دسته بندی پست را وارد نمائید",
            'image.required' => 'تصویر به درستی بارگزاری نشده است',
            'image.image' => 'تصویر به درستی بارگزاری نشده است',
            'title.required' => 'عنوان را وارد کنید ',
        ]);
    }

    private function validateupdateform(Post $post)
    {
        request()->validate([

            'description' => ['required'],
            'tags' => ['required'],
            'categories' => ['required'],
            'status' => ['required'],
            'title' => ['required'],

        ], [

            'description.required' => "توضیحات پست را وارد نمائید",
            'tags.required' => "  تگ پست را وارد نمائید",
            'categories.required' => "  دسته بندی پست را وارد نمائید",
            'status.required' => "  وضعیت را انتخاب کنید",
            'title.required' => 'عنوان را وارد کنید ',
        ]);
    }

    private function getData(): array
    {
        $data = [
            'description' => request('description'),
            'status' => request('status'),
            'title' => request('title'),
        ];

        if (request()->hasFile('image')) {
            $data['image'] = $this->uploadFile(request()->file('image'), Constant::POSTS_AVATAR_PATH);
        }
        return $data;
    }


}
