<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Constant;
use App\Filters\UsersFilter;
use App\Models\Category;
use App\Models\SocialNetwork;
use App\Models\SocialNetworkUser;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UsersController extends BaseController
{
    public function index()
    {
        $users = User::filter(new UsersFilter())->paginate();
        return view('admin.users.all', compact('users'));
    }

    public function create()
    {
        $activityStatuses = Constant::getStatusesViewer();
        return view('admin.users.create', compact('activityStatuses'));
    }

    public function store()
    {

        // validate request
        $this->validateStoreUser();

        // prepare data to insert in db
        $data = $this->getData();

        // create new user
        $user = User::create($data);

        // redirect
        if ($user instanceof User) {
            return redirect()->route('admin.users.all')->with('success', 'ثبت کاربر با موفقیت انجام شد');
        }
        return redirect()->back();
    }

    public function show(User $user)
    {

        $activityStatuses = Constant::getStatusesViewer();
        return view('admin.users.show', compact('user', 'activityStatuses'));
    }

    public function update(Request $request, User $user)
    {


        $this->validateupdateform($user);


        $data = $this->getData();

        $result = $user->update($data);

        if ($result) {

            return redirect()->route('admin.users.show', $user)->with('success', 'اطلاعات کاربر با موفقیت ویرایش شد');
        }
        return redirect()->back();
    }

    public function delete(User $user)
    {
        $user->delete();
        return redirect()->back();
    }

    public function comments(User $user)
    {
        $comments = $user->comments;
        return view('admin.users.comments', compact('comments', 'user'));
    }

    public function posts(User $user)
    {

        $posts = $user->posts;
        return view('admin.users.posts', compact('posts', 'user'));
    }

    public function subscriptions(User $user)
    {
        $subscriptions= $user->subscriptions;
        return view('admin.users.subscriptions',compact('user','subscriptions'));

    }
    public function searchWithAjax(): \Illuminate\Http\JsonResponse
    {
        $q = request('q');

        $users = User::select('id', DB::raw("CONCAT(users.first_name,' ',users.last_name) as name"))
            ->where('last_name', 'like', "%$q%")
            ->orWhere('first_name', 'like', "%$q%")
            ->get();
        return response()->json(['data' => $users]);
    }

    private function validateStoreUser()
    {

        request()->validate([
            'first_name' => ['required'],
            'last_name' => ['required'],
            'mobile' => ['required', 'unique:users'],
            'email' => ['nullable', 'email', 'unique:users'],
            'password' => ['required', 'confirmed'],
            'avatar' => ['nullable', 'image'],
            'status' => ['required', Rule::in([Constant::ACTIVE, Constant::IN_ACTIVE])],
        ], [
            'first_name.required' => "نام خود را وارد نمائید",
            'last_name.required' => "نام خانوادگی خود را وارد نمائید",
            'mobile.required' => "شماره موبایل خود را وارد نمائید",
            'mobile.unique' => "شماره موبایل قبلا ثبت شده است ",
            'username.required' => "نام کاربری خود را وارد نمائید",
            'password.required' => "رمز عبور خود را وارد نمائید",
            'password.confirmed' => 'رمز عبور با تکرار آن مطابقت ندارد   ',
            'email.email' => 'ایمیل وارد شده معتبر نمیباشد',
            'email.unique' => 'ایمیل وارد شده قبلا ثبت شده است',
            'status.in' => 'وضعیت کاربر به درستی انتخاب نشده است',
        ]);

    }

    private function validateupdateform(User $user)
    {
        request()->validate([
            'first_name' => ['required'],
            'last_name' => ['required'],
            'mobile' => ['required', Rule::unique('users', 'mobile')->ignore($user)],
            'email' => ['nullable', 'email', Rule::unique('users', 'email')->ignore($user)],
            'avatar' => ['nullable', 'image'],
            'status' => ['required', Rule::in([Constant::ACTIVE, Constant::IN_ACTIVE])],
        ], [
            'first_name.required' => "نام خود را وارد نمائید",
            'last_name.required' => "نام خانوادگی خود را وارد نمائید",
            'mobile.required' => "شماره موبایل خود را وارد نمائید",
            'mobile.unique' => "شماره موبایل قبلا ثبت شده است ",
            'username.required' => "نام کاربری خود را وارد نمائید",
            'email.email' => 'ایمیل وارد شده معتبر نمیباشد',
            'email.unique' => 'ایمیل وارد شده قبلا ثبت شده است',
            'status.in' => 'وضعیت کاربر به درستی انتخاب نشده است',
        ]);
    }

    private function getData(): array
    {
        $data = [
            'first_name' => request('first_name'),
            'last_name' => request('last_name'),
            'mobile' => request('mobile'),
            'phone' => request('phone'),
            'email' => request('email'),
            'password' => Hash::make(request('password')),
            'status' => request('status'),
        ];

        if (request()->hasFile('avatar')) {
            $data['avatar'] = $this->uploadFile(request()->file('avatar'), Constant::USERS_AVATAR_PATH);
        }
        return $data;
    }


}
